// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : entity.cpp
// Date           : February 2008
//
// Description    : Refer to description in corresponding header.
//
// ///////////////////////////////////////////////////////////////////////////


#include "entity.h"
#include "Level.h"
#include "Application.h"

#include "EntityPlayer.h"
#include "EntityBug.h"
#include "EntityCar.h"
#include "EntityTokenCollector.h"
#include "EntityLaser.h"
#include "EntityEye.h"
#include "EntityThrownToken.h"
#include "EntityAnomaly.h"
#include "EntityShot.h"
#include "EntityPlatform.h"
#include "EntityParticle.h"
#include "EntityCannon.h"
#include "EntitySwitch.h"
#include "EntityFish.h"
#include "EntitySub.h"
#include "EntitySubShot.h"
#include "EntityFly.h"
#include "EntityShodan.h"



Entity::Entity() :
  m_Direction( Dobbs::DIR_LEFT ),
  m_SpawnPosX( 0 ),
  m_SpawnPosY( 0 ),
  m_RemoveMe( false ),
  m_Hidden( false ),
  m_LifeTime( 0.0f ),
  m_Fadeout( false ),
  m_CurrentAnimation( Dobbs::ANIM_NONE ),
  m_AnimationFrame( 0 ),
  m_AnimationFrameTime( 0.0f ),
  m_UsingAngle( false ),
  m_Color( 0xffffffff ),
  m_Angle( 0.0f ),
  m_FractX( 0.0f ),
  m_FractY( 0.0f ),
  m_OnGround( false ),
  m_FallSpeed( 0 ),
  m_FallCount( 0 ),
  m_Jumping( false ),
  m_JumpVelocity( 0.0f ),
  m_MovedLastFrame( false ),
  m_Floating( false ),
  m_pCarriedBy( NULL ),
  m_pPreviousCarrier( NULL ),
  m_CarryAgainDelay( 0.0f ),
  m_pLevel( NULL ),
  m_X( 0 ),
  m_Y( 0 ),
  m_Z( 0.9999f ),
  m_ExtraData( 0 ),
  m_ExtraData2( 0 ),
  m_Sleeping( false ),
  m_State( 0 ),
  m_StateTime( 0.0f ),
  m_PickupDelay( 0.0f ),
  m_HitTime( 0.0f ),
  m_Health( 1 ),
  m_CanSleep( true ),
  m_CanBeCarried( false ),
  m_CanCollideWithBackground( true ),
  m_MoveCarriedObjectFlags( 0 ),
  m_DiesIfOutsideScreen( false )
{

  m_DisplayOffset.x = 0;
  m_DisplayOffset.y = 0;

  SetRect( &m_CollisionRect, -10, -20, 20, 20 );

  // Initialize remaining members
  m_Type            = Dobbs::ENTITY_TYPE_NONE;
  m_VelX            = 0.0;
  m_VelY            = 0.0;

  m_JumpCount       = 0;     // Jump count since last touched ground

}



Entity::~Entity() 
{

}



void Entity::SetPosition( int xVal, int yVal ) 
{

  m_X = xVal;
  m_Y = yVal;

}



int Entity::GetX() 
{ 
  return m_X; 
}



int Entity::GetY() 
{ 
  return m_Y; 
}



int Entity::GetSpawnX() 
{ 
  return m_SpawnPosX; 
}



int Entity::GetSpawnY() 
{ 
  return m_SpawnPosY; 
}



Dobbs::EntityTypes Entity::GetType() 
{ 

  return m_Type; 

}



void Entity::Unhide()
{

  m_Hidden = false;

}



void Entity::Init( bool Editor )
{
}



void Entity::SetDirection( Dobbs::DirFlags Dir )
{

  m_Direction = Dir;

}



void Entity::Update( Level& aLevel )
{

  if ( ( !m_Jumping )
  &&   ( m_pCarriedBy == NULL )
  &&   ( !m_Floating ) )
  {
    if ( aLevel.CanMove( this, 0, 1, Dobbs::DIR_FALL ) )
    {
      // start falling
      if ( m_OnGround )
      {
        m_OnGround = false;
        m_FallSpeed = 10;
      }
      m_FallCount++;
      if ( m_FallCount % 3 )
      {
        // accellerate fall
        m_FallSpeed += 30;
      }
    }
    else
    {
      if ( !m_OnGround )
      {
        m_OnGround = true;
        m_FractY    = 0.0f;
        m_JumpCount = 0;
        m_FallCount = 0;
        m_FallSpeed = 10;
        OnEvent( EE_LAND );
      }
    }
  }
  else if ( m_Jumping )
  {
    ++m_JumpCount;
    m_JumpVelocity += 40.0f;
    SetVelY( m_JumpVelocity );
    if ( m_JumpVelocity >= 0.0f )
    {
      // end of top jump arc, now it's gravity falling
      m_Jumping = false;
      m_VelY = 0.0f;
      OnEvent( EE_JUMP_DONE );
    }
  }

}



int Entity::Move( Level& aLevel, float DX, float DY, int DirFlags )
{

  if ( ( !m_CanCollideWithBackground )
  ||   ( IsHidden() ) )
  {
    // no need to check for background collision
    m_FractX += DX;
    m_FractY += DY;
    while ( m_FractX >= 1.0f )
    {
      m_X++;
      m_FractX -= 1.0f;
    }
    while ( m_FractX <= -1.0f )
    {
      m_X--;
      m_FractX += 1.0f;
    }
    while ( m_FractY >= 1.0f )
    {
      m_Y++;
      m_FractY -= 1.0f;
    }
    while ( m_FractY <= -1.0f )
    {
      m_Y--;
      m_FractY += 1.0f;
    }
    return Dobbs::BLOCKED_NONE;
  }

  int     BlockFlags = Dobbs::BLOCKED_NONE;

  m_FractX += DX;
  m_FractY += DY;

  if ( !( DirFlags & ( Dobbs::DIR_PLATFORM_MOVE | Dobbs::DIR_DIRECTED_MOVE ) ) )
  {
    m_MovedLastFrame = true;
  }

  Entity*   OldCarrier = m_pCarriedBy;

  while ( ( m_FractX >= 1.0f )
  ||      ( m_FractX < 0.0f )
  ||      ( m_FractY >= 1.0f )
  ||      ( m_FractY < 0.0f ) )
  {
    if ( m_FractY >= 1.0f )
    {
      if ( ( m_pCarriedBy != NULL )
      &&   ( OldCarrier != m_pCarriedBy )
      &&   ( aLevel.IsPlatform( m_pCarriedBy ) ) )
      {
        // blocked
        m_FractY    = 0.0f;
        m_JumpCount = 0;
        m_FallCount = 0;
        m_FallSpeed = 10;
        BlockFlags |= Dobbs::BLOCKED_BOTTOM | Dobbs::BLOCKED_Y;
        m_OnGround = true;
        OnEvent( EE_LAND );
        //dh::Log( "Land b" );
      }
      else if ( aLevel.CanMove( this, 0, 1, DirFlags ) )
      {
        aLevel.Move( this, 0, 1, DirFlags );
        m_FractY -= 1.0f;
        if ( !( DirFlags & Dobbs::DIR_PLATFORM_MOVE ) )
        {
          if ( m_OnGround )
          {
            m_OnGround = false;
          }
        }
        if ( !( DirFlags & Dobbs::DIR_FORCED_MOVE ) )
        {
          OnEvent( EE_MOVED_DOWN );
        }
      }
      else
      {
        // blocked
        m_FractY    = 0.0f;
        if ( !m_OnGround )
        {
          m_JumpCount = 0;
          m_FallCount = 0;
          m_FallSpeed = 10;
          BlockFlags |= Dobbs::BLOCKED_BOTTOM | Dobbs::BLOCKED_Y;
          m_OnGround = true;
          OnEvent( EE_LAND );
          //dh::Log( "Land c" );
        }
      }
    }
    else if ( m_FractY < 0.0f )
    {
      if ( aLevel.CanMove( this, 0, -1, DirFlags ) )
      {
        aLevel.Move( this, 0, -1, DirFlags );
        m_FractY += 1.0f;
        if ( !( DirFlags & Dobbs::DIR_FORCED_MOVE ) )
        {
          OnEvent( EE_MOVED_UP );
        }
      }
      else
      {
        // blocked
        if ( m_Jumping )
        {
          m_Jumping   = false;
          m_JumpCount = 0;
          m_VelY      = 0.0f;
        }
        m_FractY = 0.0f;
        BlockFlags |= Dobbs::BLOCKED_TOP | Dobbs::BLOCKED_Y;
        OnEvent( EE_BLOCKED_TOP );
      }
    }
    if ( m_FractX >= 1.0f )
    {
      if ( aLevel.CanMove( this, 1, 0, DirFlags ) )
      {
        aLevel.Move( this, 1, 0, DirFlags );
        m_FractX -= 1.0f;
        if ( !( DirFlags & Dobbs::DIR_FORCED_MOVE ) )
        {
          OnEvent( EE_MOVED_RIGHT );
        }
      }
      else
      {
        // blocked
        m_FractX = 0.0f;
        BlockFlags |= Dobbs::BLOCKED_RIGHT | Dobbs::BLOCKED_X;
        OnEvent( EE_BLOCKED_RIGHT );
      }
    }
    else if ( m_FractX < 0.0f )
    {
      if ( aLevel.CanMove( this, -1, 0, DirFlags ) )
      {
        aLevel.Move( this, -1, 0, DirFlags );
        m_FractX += 1.0f;
        if ( !( DirFlags & Dobbs::DIR_FORCED_MOVE ) )
        {
          OnEvent( EE_MOVED_LEFT );
        }
      }
      else
      {
        // blocked
        m_FractX = 0.0f;
        BlockFlags |= Dobbs::BLOCKED_LEFT | Dobbs::BLOCKED_X;
        OnEvent( EE_BLOCKED_LEFT );
      }
    }
    if ( !( DirFlags & Dobbs::DIR_PLATFORM_MOVE ) )
    {
      aLevel.CheckPlatformBelow( this );
    }
    if ( aLevel.IsPlatform( this ) )
    {
      // if we are a platform check other carriable entites, we might be moving through them in one frame otherwise
      std::list<Entity*>::iterator    itE( aLevel.m_Entities.begin() );
      while ( itE != aLevel.m_Entities.end() )
      {
        Entity*   pOtherEntity = *itE;

        if ( pOtherEntity != this )
        {
          Entity*   pOldCarrier2 = pOtherEntity->Carrier();
          aLevel.CheckPlatformBelow( pOtherEntity );

          if ( pOldCarrier2 != pOtherEntity->Carrier() )
          {
            pOtherEntity->m_FractY    = 0.0f;
            if ( !pOtherEntity->m_OnGround )
            {
              pOtherEntity->m_JumpCount = 0;
              pOtherEntity->m_FallCount = 0;
              pOtherEntity->m_FallSpeed = 10;
              pOtherEntity->m_OnGround = true;
              pOtherEntity->OnEvent( EE_LAND );
              //dh::Log( "Land a" );
            }
          }
        }

        ++itE;
      }
    }
  }
  // check once more after moving
  if ( !( DirFlags & Dobbs::DIR_PLATFORM_MOVE ) )
  {
    aLevel.CheckPlatformBelow( this );
  }
  if ( aLevel.IsPlatform( this ) )
  {
    // if we are a platform check other carriable entites, we might be moving through them in one frame otherwise
    std::list<Entity*>::iterator    itE( aLevel.m_Entities.begin() );
    while ( itE != aLevel.m_Entities.end() )
    {
      Entity*   pOtherEntity = *itE;

      if ( pOtherEntity != this )
      {
        Entity*   pOldCarrier2 = pOtherEntity->Carrier();
        aLevel.CheckPlatformBelow( pOtherEntity );

        if ( pOldCarrier2 != pOtherEntity->Carrier() )
        {
          pOtherEntity->m_FractY    = 0.0f;
          if ( !pOtherEntity->m_OnGround )
          {
            pOtherEntity->m_JumpCount = 0;
            pOtherEntity->m_FallCount = 0;
            pOtherEntity->m_FallSpeed = 10;
            pOtherEntity->m_OnGround = true;
            pOtherEntity->OnEvent( EE_LAND );
            //dh::Log( "Land d" );
          }
        }
      }

      ++itE;
    }
  }
  return BlockFlags;

}



void Entity::UpdateAnimation( float ElapsedTime )
{

  if ( m_CurrentAnimation != Dobbs::ANIM_NONE )
  {
    m_AnimationFrameTime += ElapsedTime;

    Dobbs::Animation&   Anim( g_App.m_Animations[m_CurrentAnimation] );

    while ( m_AnimationFrameTime >= Anim.FrameSpeed[m_AnimationFrame] )
    {
      m_AnimationFrameTime -= Anim.FrameSpeed[m_AnimationFrame];

      m_AnimationFrame++;
      if ( m_AnimationFrame >= Anim.Sections.size() )
      {
        m_AnimationFrame = 0;
      }
    }
  }

}



void Entity::UpdateTimed( Level& aLevel, float ElapsedTime )
{

  m_StateTime += ElapsedTime;

  if ( m_HitTime > 0.0f )
  {
    m_HitTime -= ElapsedTime;
    if ( m_HitTime < 0.0f )
    {
      m_HitTime = 0.0f;
    }
  }

  if ( m_PickupDelay > 0.0f )
  {
    m_PickupDelay -= ElapsedTime;
    if ( m_PickupDelay < 0.0f )
    {
      m_PickupDelay = 0.0f;
    }
  }

  if ( m_pCarriedBy == NULL )
  {
    if ( m_CarryAgainDelay > 0.0f )
    {
      m_CarryAgainDelay -= ElapsedTime;
      if ( m_CarryAgainDelay < 0.0f )
      {
        m_CarryAgainDelay = 0.0f;
        m_pPreviousCarrier = NULL;
      }
    }
  }

  UpdateAnimation( ElapsedTime );

  if ( ( !m_OnGround )
  &&   ( !m_Floating )
  &&   ( m_pCarriedBy == NULL )
  &&   ( !m_Jumping ) )
  {
    Move( aLevel, 0.0f, m_FallSpeed * ElapsedTime, Dobbs::DIR_FALL );
  }
  if ( ( m_VelX != 0.0f )
  ||   ( m_VelY != 0.0f ) )
  {
    int   DirFlags = Dobbs::DIR_NONE;
    if ( m_VelX < 0.0f )
    {
      DirFlags |= Dobbs::DIR_LEFT;
    }
    else if ( m_VelX > 0.0f )
    {
      DirFlags |= Dobbs::DIR_RIGHT;
    }
    if ( m_VelY < 0.0f )
    {
      DirFlags |= Dobbs::DIR_UP;
    }
    else if ( m_VelY > 0.0f )
    {
      DirFlags |= Dobbs::DIR_DOWN;
    }
    Move( aLevel, m_VelX * ElapsedTime, m_VelY * ElapsedTime, DirFlags );
  }

  if ( m_Fadeout )
  {
    m_LifeTime += ElapsedTime;
    if ( m_LifeTime >= 1.0f )
    {
      Die();
      m_LifeTime = 1.0f;
    }
    SetAlpha( (int)( ( 1.0f - m_LifeTime ) * 255 ) );
  }
  else if ( m_LifeTime > 0.0f )
  {
    m_LifeTime -= ElapsedTime;
    if ( m_LifeTime <= 0.0f )
    {
      Die();
    }
  }

  HandleAnimation();
  m_MovedLastFrame = false;

}



bool Entity::CanBeRemoved() const
{

  return m_RemoveMe;

}



bool Entity::IsHidden()
{

  return m_Hidden;

}



void Entity::Hide()
{

  m_Hidden = true;

}



void Entity::Fadeout()
{

  if ( !m_Fadeout )
  {
    m_Fadeout = true;
    m_LifeTime = 0.0f;
  }
}



void Entity::SetAlpha( int Alpha )
{

  if ( Alpha < 0 )
  {
    Alpha = 0;
  }
  if ( Alpha > 255 )
  {
    Alpha = 255;
  }
  m_Color = ( m_Color & 0x00ffffff ) | ( Alpha << 24 );

}



bool Entity::CollidesWith( Entity* pOtherEntity )
{

  if ( ( IsHidden() )
  ||   ( pOtherEntity->IsHidden() ) )
  {
    return false;
  }

  RECT    rcTemp;

  return !!IntersectRect( &rcTemp, &CollisionRect(), &pOtherEntity->CollisionRect() );

}



RECT Entity::CollisionRect()
{

  RECT    Coll( m_CollisionRect );

  OffsetRect( &Coll, m_X, m_Y );
  Coll.right++;
  Coll.bottom++;

  return Coll;

}



void Entity::Render( int XOffset, int YOffset )
{

  if ( IsHidden() )
  {
    return;
  }

  Dobbs::Animation&   Anim( g_App.m_Animations[m_CurrentAnimation] );

  if ( Anim.Sections.empty() )
  {
    return;
  }

  if ( m_AnimationFrame >= Anim.Sections.size() )
  {
    m_AnimationFrame = 0;
  }

  const TextureSection& TexSec( Anim.Sections[m_AnimationFrame] );

  if ( ( -XOffset + m_X < -140 )
  ||   ( -XOffset + m_X > Dobbs::SCREEN_WIDTH + 100 )
  ||   ( -YOffset + m_Y < -140 )
  ||   ( -YOffset + m_Y > Dobbs::SCREEN_HEIGHT + 100 ) )
  {
    // skip tiles that are outside the visible screen
  }
  else if ( m_UsingAngle )
  {
    g_App.RenderTextureSectionRotated( TexSec, -XOffset + m_X, -YOffset + m_Y, m_Angle );
  }
  else
  {
    if ( m_HitTime > 0.0f )
    {
      // display whitened
      g_App.RenderBatchedTiles();
      g_App.RenderTextureSectionWhitened( TexSec, -XOffset + m_X - m_DisplayOffset.x, -YOffset + m_Y - m_DisplayOffset.y, m_Color, m_Z );
    }
    else
    {
      g_App.RenderTextureSection( TexSec, -XOffset + m_X - m_DisplayOffset.x, -YOffset + m_Y - m_DisplayOffset.y, m_Color, m_Z );
    }
  }

}



void Entity::SetColor( unsigned int Color )
{

  m_Color = Color;

}



void Entity::SetFrame( Dobbs::AnimationIndex Frame )
{

  if ( m_CurrentAnimation != Frame )
  {
    m_CurrentAnimation    = Frame;
    m_AnimationFrame      = 0;
    m_AnimationFrameTime  = 0.0f;
  }

}



bool Entity::Jump( Level& aLevel, int JumpPower ) 
{
  // Ignore jump if already jumped twice
  if ( m_JumpCount >= 1 )
  {
    return false;
  }
  if ( !m_OnGround )
  {
    return false;
  }
  return ForceJump( aLevel, JumpPower );

}



bool Entity::ForceJump( Level& aLevel, int JumpPower ) 
{
  if ( m_pCarriedBy )
  {
    // auto-drop off vehicle
    m_pCarriedBy->DropOff( this );
    m_CarryAgainDelay = 0.5f;
  }

  m_Jumping = true;

  // Increment jump counter (to ensure max is not exceeded)
  m_JumpCount++;

  // Deassert onGround flag
  m_OnGround = false;
  m_FallSpeed = 10;

  // Initiate jump with max upward velocity
  m_JumpVelocity = -(float)JumpPower;

  //aLevel.JumpEffect( m_X, m_Y );
  OnEvent( EE_JUMP );
  return true;

}



Dobbs::DirFlags Entity::Direction() const
{
  
  return m_Direction;

}



void Entity::SetVelX( float VelX )
{

  m_VelX = VelX;
  if ( m_VelX > 0.0f )
  {
    m_Direction = Dobbs::DIR_RIGHT;
  }
  else if ( m_VelX < 0.0f )
  {
    m_Direction = Dobbs::DIR_LEFT;
  }

}



void Entity::SetLifeTime( float LifeTime )
{

  m_LifeTime = LifeTime;

}



void Entity::SetVelY( float VelY )
{

  m_VelY = VelY;

}



float Entity::GetVelX()
{

  return m_VelX;

}



float Entity::GetVelY()
{

  return m_VelY;

}



void Entity::HandleAnimation()
{
}



bool Entity::OnGround() const
{

  return m_OnGround;

}



Entity* Entity::SpawnEntity( Dobbs::EntityTypes EType, int X, int Y, Dobbs::DirFlags StartDir )
{

  Entity*   pEntity = NULL;

  switch ( EType )
  {
    case Dobbs::ENTITY_TYPE_SECURITY_PASS:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_SECURITY_PASS );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_SHODAN:
      pEntity = new EntityShodan();
      break;
    case Dobbs::ENTITY_TYPE_SUB:
      pEntity = new EntitySub();
      break;
    case Dobbs::ENTITY_TYPE_FLY:
      pEntity = new EntityFly();
      break;
    case Dobbs::ENTITY_TYPE_PLAYER:
      pEntity = new EntityPlayer();
      break;
    case Dobbs::ENTITY_TYPE_TOKEN:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_TOKEN_ROTATE );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_THROWN_TOKEN:
      pEntity = new EntityThrownToken();
      break;
    case Dobbs::ENTITY_TYPE_EXIT:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_EXIT_SIGN );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_KEY_RED:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_KEY_RED );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_KEY_GREEN:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_KEY_GREEN );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_KEY_BLUE:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_KEY_BLUE );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_KEY_YELLOW:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_KEY_YELLOW );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_DOOR:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_DOOR );
      SetRect( &pEntity->m_CollisionRect, 0, 0, 64, 64 );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      pEntity->m_Z = 0.99992f;
      break;
    case Dobbs::ENTITY_TYPE_DOOR_LOCKED:
      pEntity = new Entity();
      pEntity->SetFrame( Dobbs::ANIM_DOOR_LOCKED );
      SetRect( &pEntity->m_CollisionRect, 0, 0, 64, 64 );
      pEntity->m_Floating = true;
      pEntity->m_CanCollideWithBackground = false;
      pEntity->m_Z = 0.99992f;
      break;
    case Dobbs::ENTITY_TYPE_BUG_L:
    case Dobbs::ENTITY_TYPE_BUG_R:
      pEntity = new EntityBug( EType );
      break;
    case Dobbs::ENTITY_TYPE_CAR:
      pEntity = new EntityCar();
      break;
    case Dobbs::ENTITY_TYPE_SWITCH:
      pEntity = new EntitySwitch();
      break;
    case Dobbs::ENTITY_TYPE_TOKEN_COLLECTOR:
      pEntity = new EntityTokenCollector();
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_LASER_D:
    case Dobbs::ENTITY_TYPE_LASER_U:
    case Dobbs::ENTITY_TYPE_LASER_L:
    case Dobbs::ENTITY_TYPE_LASER_R:
      pEntity = new EntityLaser( EType );
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_LASER_H:
      pEntity = new EntityLaser( EType );
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_LASER_H );
      SetRect( &pEntity->m_CollisionRect, 0, 10, 32, 12 );
      break;
    case Dobbs::ENTITY_TYPE_LASER_V:
      pEntity = new EntityLaser( EType );
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_LASER_V );
      SetRect( &pEntity->m_CollisionRect, 10, 0, 12, 32 );
      break;
    case Dobbs::ENTITY_TYPE_PLAYER_EXPLOSION:
      pEntity = new Entity();
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_EXPLOSION );
      pEntity->m_LifeTime = 0.799f;
      SetRect( &pEntity->m_CollisionRect, 0, 0, 256, 256 );
      pEntity->m_DisplayOffset.x = 128;
      pEntity->m_DisplayOffset.y = 128;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_EXPLOSION:
      pEntity = new Entity();
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_ENEMY_EXPLODE );
      pEntity->m_LifeTime = 0.399f;
      SetRect( &pEntity->m_CollisionRect, 0, 0, 256, 256 );
      pEntity->m_DisplayOffset.x = 128;
      pEntity->m_DisplayOffset.y = 128;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_EYE:
      pEntity = new EntityEye();
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_ANOMALY:
      pEntity = new EntityAnomaly();
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_ANOMALY );
      SetRect( &pEntity->m_CollisionRect, -48, -48, 48, 48 );
      pEntity->m_DisplayOffset.x = 64;
      pEntity->m_DisplayOffset.y = 64;
      pEntity->m_Health = 10;
      pEntity->m_CanCollideWithBackground = false;
      break;
    case Dobbs::ENTITY_TYPE_SHOT:
      pEntity = new EntityShot();
      pEntity->m_Floating = true;
      pEntity->SetFrame( Dobbs::ANIM_SHOT );
      SetRect( &pEntity->m_CollisionRect, -6, -6, 6, 6 );
      pEntity->m_DisplayOffset.x = 8;
      pEntity->m_DisplayOffset.y = 8;
      pEntity->m_Health = 1;
      break;
    case Dobbs::ENTITY_TYPE_PLATFORM:
      pEntity = new EntityPlatform();
      break;
    case Dobbs::ENTITY_TYPE_PARTICLE_EXPLOSION:
      pEntity = new EntityParticle();
      pEntity->SetFrame( Dobbs::ANIM_EXPLOSION_PARTICLE_BIG );
      pEntity->m_Color = 0xffffffff;
      pEntity->m_LifeTime = 0.6f;
      pEntity->m_CanCollideWithBackground = false;
      pEntity->m_Z = 0.99992f;
      break;
    case Dobbs::ENTITY_TYPE_CANNON:
      pEntity = new EntityCannon();
      break;
    case Dobbs::ENTITY_TYPE_FISH_L:
    case Dobbs::ENTITY_TYPE_FISH_R:
      pEntity = new EntityFish( EType );
      break;
    case Dobbs::ENTITY_TYPE_SUB_SHOT:
      pEntity = new EntitySubShot();
      break;
    default:
      return NULL;
  }
  pEntity->m_Type = EType;
  pEntity->SetPosition( X, Y );
  pEntity->SetDirection( StartDir );
  return pEntity;

}



void Entity::OnEvent( const EntityEvent Event, int Param1, int Param2, const std::string& TextParam, Entity* pEntity )
{

  switch ( Event )
  {
    case EE_CARRY:
      break;
    case EE_DIE:
      if ( m_pCarriedBy )
      {
        m_pCarriedBy->DropOff( this );
      }
      break;
    case EE_KILLED:
      m_pLevel->SpawnEntity( Dobbs::ENTITY_TYPE_EXPLOSION, 0, GetX(), GetY() );
      g_App.PlaySound( "Player.Explode" );
      break;
    case EE_INIT:
      m_SpawnPosX = m_X;
      m_SpawnPosY = m_Y;
      if ( m_Type == Dobbs::ENTITY_TYPE_SECURITY_PASS )
      {
        switch ( m_ExtraData )
        {
          case 0:
            m_Color = 0xffe0ef00;
            break;
          case 1:
            m_Color = 0xffdf3000;
            break;
          case 2:
            m_Color = 0xff00a4ef;
            break;
          case 3:
            m_Color = 0xff65c100;
            break;
        }
      }
      break;
  }

}



void Entity::Die()
{

  m_RemoveMe = true;

}



void Entity::Carry( Entity* pCarried )
{

  if ( ( pCarried->m_CarryAgainDelay > 0.0f )
  &&   ( pCarried->m_pPreviousCarrier == this ) )
  {
    // this entity cannot be carried by this carrier yet
    return;
  }

  if ( ( pCarried->m_pCarriedBy != this )
  &&   ( pCarried->m_pCarriedBy != NULL ) )
  {
    pCarried->m_pCarriedBy->DropOff( pCarried );
  }
  if ( pCarried->m_pCarriedBy == NULL )
  {
    pCarried->m_pCarriedBy = this;
    m_CarriedEntities.push_back( pCarried );

    OnEvent( EE_CARRY, 0, 0, std::string(), pCarried );
    pCarried->OnEvent( EE_CARRIED, 0, 0, std::string(), this );

    if ( pCarried->m_pCarriedBy )
    {
      pCarried->m_JumpCount = 0;
    }
  }

}



void Entity::DropOff( Entity* pCarried )
{

  if ( pCarried->m_pCarriedBy == this )
  {
    pCarried->m_pPreviousCarrier = this;
    pCarried->m_pCarriedBy = NULL;
    m_CarriedEntities.remove( pCarried );

    OnEvent( EE_DROP, 0, 0, std::string(), pCarried );
    pCarried->OnEvent( EE_DROPPED, 0, 0, std::string(), this );
  }

}



void Entity::SetState( int NewState )
{

  if ( m_State != NewState )
  {
    m_State = NewState;
    m_StateTime = 0.0f;
  }

}



bool Entity::CanBePicked() const
{

  return ( m_PickupDelay == 0.0f );

}



bool Entity::Hit()
{

  m_HitTime = 0.1f;
  return true;

}



void Entity::Hurt( int Damage )
{

  if ( Damage >= m_Health )
  {
    m_Health = 0;
    Die();
    OnEvent( EE_KILLED );
  }
  else
  {
    m_Health -= Damage;
    OnEvent( EE_HURT );
  }

}



Entity* Entity::Carrier() const
{

  return m_pCarriedBy;

}



bool Entity::CanBeCarried() const
{

  if ( m_Jumping )
  {
    return false;
  }
  return m_CanBeCarried;

}



float Entity::Angle() const
{

  return m_Angle;

}



void Entity::SetAngle( float Angle )
{

  m_Angle = Angle;

}



int Entity::MoveCarriedObjectFlags() const
{

  return m_MoveCarriedObjectFlags;

}